/*
  ==================================================================================
  ==                       360° NEOPIXEL DIGITAL TRIPWIRE                         ==
  ==================================================================================
  == WIRING SCHEMATIC (Arduino Nano & TCA9548A Multiplexer)                       ==
  == ---------------------------------------------------------------------------- ==
  == CRITICAL NOTE: The NeoPixel Ring MUST be powered by a source power.          ==
  == The Arduino's GND and the external supply's GND MUST be connected.           ==
  == ---------------------------------------------------------------------------- ==
  == 1. TCA9548A Multiplexer to Arduino Nano:                                     ==
  ==   - VCC -> Nano 5V                                                           ==
  ==   - GND -> Nano GND                                                          ==
  ==   - SCL -> Nano A5                                                           ==
  ==   - SDA -> Nano A4                                                           ==
  ==                                                                              ==
  == 2. VL53L0X Sensors to Multiplexer (Repeat for all 8 sensors):                ==
  ==   - Sensor 0 VCC/GND -> 5V/GND Rails                                         ==
  ==   - Sensor 0 SCL/SDA -> Multiplexer SC0/SD0                                  ==
  ==   - Sensor 1 SCL/SDA -> Multiplexer SC1/SD1                                  ==
  ==   - ...and so on for all 8 channels.                                         ==
  ==                                                                              ==
  == 3. NeoPixel Ring:                                                            ==
  ==   - Power (5V) -> EXTERNAL 5V Power Supply (+)                               ==
  ==   - GND        -> EXTERNAL 5V Power Supply (-) AND Nano GND (common ground)  ==
  ==   - Data In    -> Nano D6                                                    ==
  ==                                                                              ==
  == 4. Potentiometer & Buzzer:                                                   ==
  ==   - Potentiometer Middle Pin -> Nano A0 (Outer pins to 5V & GND)             ==
  ==   - Buzzer Positive (+)      -> Nano D5  (Negative to GND)                   ==
  ==================================================================================
*/
#include <Wire.h>
#include "Adafruit_VL53L0X.h"
#include "Adafruit_NeoPixel.h"

// --- Pin Definitions ---
#define NEOPIXEL_PIN 6
const int POT_PIN = A0;
const int BUZZER_PIN = 5;

// --- Configuration ---
const int NUM_SENSORS = 8;
const int NUM_PIXELS = 16; // Using a 16-pixel ring (2 pixels per sensor)
const int MIN_RANGE_MM = 50;
const int MAX_RANGE_MM = 1000;
const int TRIGGER_COOLDOWN_MS = 3000; // How long a zone stays red after trigger

// --- Objects ---
Adafruit_VL53L0X lox = Adafruit_VL53L0X();
Adafruit_NeoPixel pixels(NUM_PIXELS, NEOPIXEL_PIN, NEO_GRB + NEO_KHZ800);

// --- Global State Variables ---
int tripwireDistance = MIN_RANGE_MM;
bool isTriggered[NUM_SENSORS] = {false};
unsigned long triggerStopTime[NUM_SENSORS] = {0}; // Timestamp for cooldown

// For idle animation
int chasePosition = 0;
unsigned long lastChaseTime = 0;

// Function to select I2C bus on the multiplexer
void tcaSelect(uint8_t i) {
  if (i > 7) return;
  Wire.beginTransmission(0x70); // TCA9548A address
  Wire.write(1 << i);
  Wire.endTransmission();
}

void setup() {
  Serial.begin(115200);
  Wire.begin();

  pixels.begin();
  pixels.setBrightness(50); // Set brightness to a safe level

  // Initialize all 8 sensors
  for (uint8_t i = 0; i < NUM_SENSORS; i++) {
    tcaSelect(i);
    if (!lox.begin()) {
      Serial.print(F("Failed to boot VL53L0X on channel "));
      Serial.println(i);
    }
  }
  Serial.println("360 Tripwire Armed!");
}

void loop() {
  // 1. Set global range from potentiometer
  tripwireDistance = map(analogRead(POT_PIN), 0, 1023, MIN_RANGE_MM, MAX_RANGE_MM);

  bool systemIsAlert = false;

  // 2. Poll all sensors
  for (uint8_t i = 0; i < NUM_SENSORS; i++) {
    tcaSelect(i);
    VL53L0X_RangingMeasurementData_t measure;
    lox.rangingTest(&measure, false);

    // Check for a trigger
    if (measure.RangeStatus != 4 && measure.RangeMilliMeter < tripwireDistance) {
      isTriggered[i] = true;
      triggerStopTime[i] = 0; // Reset cooldown timer
      systemIsAlert = true;
    } else {
      // If it was triggered but now isn't, start the cooldown
      if (isTriggered[i]) {
        isTriggered[i] = false;
        triggerStopTime[i] = millis(); // Start cooldown timer
      }
    }
  }

  // 3. Update Buzzer
  if (systemIsAlert) {
    tone(BUZZER_PIN, 1200, 50); // Play a high-pitch tone
  } else {
    noTone(BUZZER_PIN);
  }
  
  // 4. Update NeoPixels
  updateNeopixels();

  delay(50); // Short delay to prevent overwhelming the I2C bus
}

void updateNeopixels() {
  // A. Idle State: Chasing blue light
  if (millis() - lastChaseTime > 75) {
    pixels.clear(); // Clear previous chase pixel
    chasePosition = (chasePosition + 1) % NUM_PIXELS;
    lastChaseTime = millis();
  }
  pixels.setPixelColor(chasePosition, pixels.Color(0, 0, 80)); // Dim blue

  // B. Override with trigger/cooldown states
  for (int i = 0; i < NUM_SENSORS; i++) {
    int pixelIndex1 = i * 2;
    int pixelIndex2 = i * 2 + 1;

    if (isTriggered[i]) {
      // Flashing red for active trigger
      uint32_t red = (millis() % 200 > 100) ? pixels.Color(255, 0, 0) : pixels.Color(0, 0, 0);
      pixels.setPixelColor(pixelIndex1, red);
      pixels.setPixelColor(pixelIndex2, red);
    } else if (triggerStopTime[i] != 0) {
      // Solid red for cooldown
      if (millis() - triggerStopTime[i] < TRIGGER_COOLDOWN_MS) {
        pixels.setPixelColor(pixelIndex1, pixels.Color(150, 0, 0));
        pixels.setPixelColor(pixelIndex2, pixels.Color(150, 0, 0));
      } else {
        triggerStopTime[i] = 0; // Cooldown finished
      }
    }
  }

  pixels.show();
}