/*
  Project: UltraSonic trip wire, Visual LED Strip range detection with buzzer.
  Description: Displays the distance measured by an I2C ultrasonic sensor on a NeoPixel strip.
  Features a red tracer with a fading tail, a blue breathing idle effect, a 
  parking-sensor style buzzer, and a potentiometer for calibration.

  ---------------------------------------------------------------------------------
  WIRING CONNECTIONS:
  ---------------------------------------------------------------------------------

  --- I2C Ultrasonic Sensor ---
  VCC -> Arduino 5V
  GND -> Arduino GND
  SDA -> Arduino A4
  SCL -> Arduino A5

  --- NeoPixel Strip (WS2812B) ---
  DIN -> 470 Ohm Resistor -> Arduino D6
  5V  -> External 5V Power Supply (+)
  GND -> External 5V Power Supply (-) AND Arduino GND (Grounds MUST be connected)
  ** IMPORTANT: Power the strip from an external 5V supply, not the Arduino 5V pin.
  ** A 1000uF capacitor between 5V and GND of the external supply is recommended.

  --- Piezo Buzzer ---
  +   -> Arduino D9
  -   -> Arduino GND

  --- 10k Potentiometer ---
  VCC (Outer Pin)       -> Arduino 5V
  GND (Other Outer Pin) -> Arduino GND
  Wiper (Middle Pin)    -> Arduino A0

  ---------------------------------------------------------------------------------
*/

#include <Wire.h>
#include <Ultrasonic.h>
#include <Adafruit_NeoPixel.h>

// -- PIN DEFINITIONS --
#define PIXEL_PIN       6  // Data pin for the NeoPixel strip
#define BUZZER_PIN      9  // Pin for the piezo buzzer
#define POT_PIN         A0 // Pin for the calibration potentiometer

// -- NEOPIXEL STRIP CONFIGURATION --
#define NUM_PIXELS      60 // Number of LEDs in your strip
Adafruit_NeoPixel strip = Adafruit_NeoPixel(NUM_PIXELS, PIXEL_PIN, NEO_GRB + NEO_KHZ800);

// -- SENSOR CONFIGURATION --
Ultrasonic ultrasonic(0); // I2C address is usually not needed for this library
#define MAX_RANGE_CM    100 // The maximum distance you want to measure (in cm)
#define MIN_RANGE_CM    5   // The minimum distance to start detection

// -- VISUAL EFFECTS CONFIGURATION --
#define FADE_RATE       15  // Speed of the red tracer fade. Higher is faster.
uint8_t redValues[NUM_PIXELS]; // Array to hold the brightness of each red pixel for fading

// -- BUZZER TIMING --
unsigned long lastBuzzerTime = 0;
int buzzerInterval = 1000; // Time between chirps in ms

void setup() {
  Serial.begin(9600);
  pinMode(BUZZER_PIN, OUTPUT);
  
  strip.begin();
  strip.setBrightness(100); // Set brightness (0-255). Start moderately.
  strip.show(); // Initialize all pixels to 'off'
}

void loop() {
  // 1. Handle Calibration
  int potValue = analogRead(POT_PIN);
  // Map potentiometer to an offset range, e.g., -10cm to +10cm
  int calibrationOffset = map(potValue, 0, 1023, -10, 10);

  // 2. Read Sensor and Apply Calibration
  long rawDistance = ultrasonic.MeasureInCentimeters();
  long distance = rawDistance + calibrationOffset;
  
  // 3. Update LEDs and Buzzer based on distance
  handleVisuals(distance);
  handleBuzzer(distance);

  delay(10); // Small delay to keep things smooth
}

void handleVisuals(long dist) {
  // First, apply the fade effect to all pixels
  for (int i = 0; i < NUM_PIXELS; i++) {
    if (redValues[i] > 0) {
      redValues[i] = max(0, redValues[i] - FADE_RATE);
    }
  }

  // Check if an object is detected within our defined range
  if (dist > MIN_RANGE_CM && dist < MAX_RANGE_CM) {
    // Map the distance to a pixel index
    int pixelIndex = map(dist, MIN_RANGE_CM, MAX_RANGE_CM, 0, NUM_PIXELS - 1);
    
    // Light up the target pixel to full red
    redValues[pixelIndex] = 255;

  } else {
    // If no object is detected, run the blue breathing effect
    breathingEffect();
  }
  
  // Finally, update the physical strip with the new colors
  for (int i = 0; i < NUM_PIXELS; i++) {
    strip.setPixelColor(i, strip.Color(redValues[i], 0, 0));
  }

  // Only show the strip *after* all colors have been set
  strip.show();
}

void breathingEffect() {
  // Use a sine wave for a smooth breathing animation
  float breath = (sin(millis() / 2000.0 * PI) + 1) / 2.0; // Oscillates between 0 and 1
  int blueValue = map(breath * 100, 0, 100, 10, 90); // Map to a dim blue range
  
  // Set all pixels to the breathing blue color, only if there's no red value
  for (int i = 0; i < NUM_PIXELS; i++) {
    if (redValues[i] == 0) {
      strip.setPixelColor(i, 0, 0, blueValue);
    }
  }
}

void handleBuzzer(long dist) {
  if (dist > MIN_RANGE_CM && dist < MAX_RANGE_CM) {
    // Make the interval shorter as the object gets closer
    buzzerInterval = map(dist, MIN_RANGE_CM, MAX_RANGE_CM, 100, 1000);
    
    if (millis() - lastBuzzerTime > buzzerInterval) {
      // Make the tone higher as the object gets closer
      int toneFreq = map(dist, MIN_RANGE_CM, MAX_RANGE_CM, 1500, 500);
      tone(BUZZER_PIN, toneFreq, 50); // Play a short 50ms chirp
      lastBuzzerTime = millis();
    }
  }
}